// Maquette Veille — Observatoire mondial des soutiens publics à l’agriculture (FARM)
// Single-file React component with Tailwind + shadcn/ui + lucide-react
// Remplacez EMBED_URL_SCOOPIT par l’URL d’intégration fournie par Scoop.it Pro.

import React, { useMemo, useState } from "react";
import { Search, Filter, Globe2, Tag as TagIcon, Calendar, Link as LinkIcon, ChevronRight, Info, ExternalLink, Leaf, TrendingUp, Scale, Building2, Newspaper, LayoutGrid, List, Download } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Switch } from "@/components/ui/switch";

// ————————————————————————————————————————————
// Types & données factices
// ————————————————————————————————————————————

type TypeSoutien = "TBP" | "TBC" | "SERV" | "MSM"; // TBP: Transferts budgétaires aux producteurs; TBC: aux consommateurs; SERV: Services collectifs; MSM: Mesures de soutien par le marché

type Objectif = "ENV" | "PROD" | "SECAL" | "EQUIT";

type Instrument =
  | "Subventions aux intrants"
  | "Paiements pour services environnementaux"
  | "Mesures de prix (MPS/tarifs)"
  | "Crédit / prêt concessionnel"
  | "Achat public / filets de sécurité";

type CarteActu = {
  id: string;
  titre: string;
  source: string;
  url: string;
  dateISO: string; // 2025-10-08
  pays: string;
  region: string;
  type: TypeSoutien;
  objectifs: Objectif[];
  instrument: Instrument;
  extrait: string;
  decryptage: string;
};

const MOCK: CarteActu[] = [
  {
    id: "1",
    titre: "Réforme des subventions aux engrais : ciblage par petites exploitations",
    source: "Ministère de l’Agriculture — Communiqué",
    url: "#",
    dateISO: "2025-10-02",
    pays: "Maroc",
    region: "Afrique du Nord",
    type: "TBP",
    objectifs: ["PROD", "EQUIT"],
    instrument: "Subventions aux intrants",
    extrait:
      "Le nouveau mécanisme introduit un ciblage par registre social et un plafonnement des volumes subventionnés par exploitation.",
    decryptage:
      "Bascule vers une aide plus ciblée, potentiellement moins distorsive. Effet attendu : gains d’efficience budgétaire et réduction des fuites.",
  },
  {
    id: "2",
    titre: "Tarifs variables sur les importations de blé pour stabiliser les prix internes",
    source: "Journal Officiel",
    url: "#",
    dateISO: "2025-09-28",
    pays: "Égypte",
    region: "Afrique du Nord",
    type: "MSM",
    objectifs: ["SECAL"],
    instrument: "Mesures de prix (MPS/tarifs)",
    extrait:
      "Mise en place d’un corridor tarifaire ajusté mensuellement en fonction des cours mondiaux.",
    decryptage:
      "Instrument de stabilisation. Impact prix → consommateurs. Risque : coût fiscal indirect et effets sur incitations à long terme.",
  },
  {
    id: "3",
    titre: "Pilote PSE pour pollinisateurs et qualité de l’eau",
    source: "Agence nationale — Note de cadrage",
    url: "#",
    dateISO: "2025-10-05",
    pays: "Sénégal",
    region: "Afrique de l’Ouest",
    type: "SERV",
    objectifs: ["ENV", "PROD"],
    instrument: "Paiements pour services environnementaux",
    extrait:
      "Contrats pluriannuels rémunèrent des pratiques agroécologiques sur 5.000 ha.",
    decryptage:
      "Signal environnemental clair (PSE). Co-bénéfices productivité si adoption de pratiques améliorées.",
  },
];

const TAG_LABELS: Record<TypeSoutien | Objectif, string> = {
  TBP: "Transferts budgétaires aux producteurs",
  TBC: "Transferts budgétaires aux consommateurs",
  SERV: "Services collectifs pour le développement du secteur",
  MSM: "Mesures de soutien par le marché",
  ENV: "Environnement",
  PROD: "Productivité & compétitivité",
  SECAL: "Sécurité alim. & nutrition",
  EQUIT: "Revenu & stabilité",
  };

const TAG_ICON: Record<TypeSoutien | Objectif, React.ReactNode> = {
  TBP: <Building2 className="h-3.5 w-3.5" />, 
  TBC: <Building2 className="h-3.5 w-3.5" />, 
  SERV: <Info className="h-3.5 w-3.5" />, 
  MSM: <Scale className="h-3.5 w-3.5" />, 
  ENV: <Leaf className="h-3.5 w-3.5" />, 
  PROD: <TrendingUp className="h-3.5 w-3.5" />, 
  SECAL: <Info className="h-3.5 w-3.5" />, 
  EQUIT: <Scale className="h-3.5 w-3.5" />, 
};

// ————————————————————————————————————————————
// Composants utilitaires
// ————————————————————————————————————————————

function Chip({
  active,
  children,
  onClick,
}: { active?: boolean; children: React.ReactNode; onClick?: () => void }) {
  return (
    <button
      onClick={onClick}
      className={`inline-flex items-center gap-1 px-3 py-1 text-xs transition ${active? 'chip shadow':'chip'}`}
    >
      {children}
    </button>
  );
}

function TagPill({ tag }: { tag: TypeSoutien | Objectif }) {
  const cls = (() => {
    switch (tag) {
      case "TBP": return "pill -tbp";
      case "TBC": return "pill -tbc";
      case "SERV": return "pill -serv";
      case "MSM": return "pill -msm";
      default: return "pill -obj"; // ENV, PROD, SECAL, EQUIT
    }
  })();
  return (
    <span className={cls}>
      {TAG_ICON[tag]}
      {TAG_LABELS[tag]}
    </span>
  );
}

function LigneMeta({ icon, children }: { icon: React.ReactNode; children: React.ReactNode }) {
  return (
    <div className="flex items-center gap-1.5 text-xs text-slate-600">
      {icon}
      <span>{children}</span>
    </div>
  );
}

// ————————————————————————————————————————————
// Carte Actu
// ————————————————————————————————————————————

function Carte({ c }: { c: CarteActu }) {
  const date = new Date(c.dateISO).toLocaleDateString("fr-FR", { day: "2-digit", month: "short", year: "numeric" });
  const typeClass = c.type==='TBP'?'pill -tbp':c.type==='TBC'?'pill -tbc':c.type==='SERV'?'pill -serv':'pill -msm';
  return (
    <Card className="group overflow-hidden rounded-2xl card-accent border border-slate-200 shadow-sm">
      <CardHeader className="pb-3">
        <div className="flex items-start justify-between gap-2">
          <CardTitle className="text-base leading-snug">
            <a href={c.url} className="inline-flex items-start gap-1 link-primary">
              {c.titre}
              <ExternalLink className="ml-1 h-4 w-4 opacity-60 group-hover:opacity-100" />
            </a>
          </CardTitle>
          <span className={`shrink-0 ${typeClass}`}>
            {TAG_ICON[c.type]}
            <span className="ml-1">{TAG_LABELS[c.type]}</span>
          </span>
        </div>
        <CardDescription className="mt-2 text-xs leading-relaxed text-slate-600">
          {c.extrait}
        </CardDescription>
      </CardHeader>
      <CardContent className="grid gap-3">
        <div className="flex flex-wrap items-center gap-3">
          <LigneMeta icon={<Globe2 className="h-4 w-4" />}>{c.pays} • {c.region}</LigneMeta>
          <LigneMeta icon={<Calendar className="h-4 w-4" />}>{date}</LigneMeta>
          <LigneMeta icon={<TagIcon className="h-4 w-4" />}>{c.instrument}</LigneMeta>
          <LigneMeta icon={<LinkIcon className="h-4 w-4" />}>{c.source}</LigneMeta>
        </div>
        <div className="flex flex-wrap gap-2">
          {c.objectifs.map((o) => (
            <TagPill key={o} tag={o} />
          ))}
        </div>
        <div className="rounded-xl border border-slate-200 bg-slate-50 p-3 text-sm leading-relaxed">
          <div className="mb-1 flex items-center gap-2 font-medium text-slate-800">
            <Info className="h-4 w-4" /> Décryptage FARM
          </div>
          <p className="text-slate-700">{c.decryptage}</p>
        </div>
        <div className="flex items-center justify-end">
          <Button variant="ghost" className="gap-1 link-primary">
            Lire la source <ChevronRight className="h-4 w-4" />
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}

// ————————————————————————————————————————————
// Section “Événements & grands rendez-vous”
// ————————————————————————————————————————————

function EvenementsRendezVous() {
  const events = [
    {
      id: "e1",
      dateISO: "2025-11-14",
      titre: "Sommet sur les soutiens publics agricoles et transition alimentaire",
      format: "Conference internationale",
      lieu: "Paris (hybride)",
      lien: "#",
      tags: ["MSM", "TBP", "EQUIT"],
      resume: "Decideurs publics, bailleurs et chercheurs debattent des reformes budgetaires et des instruments prix.",
      dossier: "SPM & tarifs",
    },
    {
      id: "e2",
      dateISO: "2025-12-03",
      titre: "Webinaire - Subventions aux intrants : ciblage et efficience",
      format: "Webinaire",
      lieu: "En ligne",
      lien: "#",
      tags: ["TBP", "PROD"],
      resume: "Retours d'experience sur registres sociaux, plafonds par exploitation et controle ex-post.",
      dossier: "Subventions aux intrants",
    },
    {
      id: "e3",
      dateISO: "2026-01-21",
      titre: "Atelier technique - PSE et indicateurs MRV",
      format: "Atelier",
      lieu: "Dakar",
      lien: "#",
      tags: ["ENV", "TBP", "MRV"],
      resume: "Conception de contrats PSE, indicateurs environnementaux et co-benefices productivite.",
      dossier: "PSE environnementaux",
    },
    {
      id: "e4",
      dateISO: "2025-11-28",
      titre: "Table ronde - Securite alimentaire et instruments de stabilisation des prix",
      format: "Table ronde",
      lieu: "Rabat (hybride)",
      lien: "#",
      tags: ["SECAL", "MSM"],
      resume: "Comparaison tarifs variables, stocks publics et transferts cibles.",
      dossier: "SPM & tarifs",
    },
  ];

  function formatDate(dateISO: string) {
    const d = new Date(dateISO);
    const day = d.toLocaleDateString("fr-FR", { day: "2-digit" });
    const month = d.toLocaleDateString("fr-FR", { month: "short" });
    const year = d.getFullYear();
    return { day, month, year };
  }

  return (
    <section className="mt-10" aria-labelledby="events-title">
      {/* Titre mis en avant (même style que les autres sections) */}
      <div className="rounded-2xl bg-gradient-to-r from-[color:var(--p-15)] to-[color:var(--s-20)] p-4 ring-1 ring-[color:var(--sand)]/50 mb-6">
        <div className="flex items-start justify-between gap-4">
          <div>
            <h2 id="events-title" className="text-2xl md:text-3xl font-extrabold tracking-tight leading-tight">
              <span className="bg-gradient-to-r from-[color:var(--p)] to-[color:var(--s)] bg-clip-text text-transparent">
                Événements & grands rendez-vous
              </span>
            </h2>
            <p className="mt-1 text-sm md:text-base text-slate-600">
              Conférences, webinaires et ateliers à venir.
            </p>
          </div>
          {/* pas d’actions */}
        </div>
        {/* Barre d’accent */}
        <div className="mt-3 h-1.5 w-28 rounded-full bg-gradient-to-r from-[color:var(--p)] to-[color:var(--s)]" />
      </div>

      {/* Cartes d'événements (inchangé) */}
      <div className="grid gap-4 md:grid-cols-2">
        {events.map((ev) => {
          const f = formatDate(ev.dateISO);
          return (
            <Card key={ev.id} className="rounded-2xl border border-slate-200 shadow-sm">
              <CardHeader className="pb-2">
                <div className="flex items-start justify-between gap-3">
                  <div>
                    <CardTitle className="text-base leading-snug text-[color:var(--p)]">{ev.titre}</CardTitle>
                    <CardDescription className="mt-1 text-xs">
                      {ev.format} • {ev.lieu}
                    </CardDescription>
                  </div>
                  <div className="shrink-0 text-center">
                    <div
                      className="rounded-xl px-2 py-1 text-xs font-medium"
                      style={{ background: "var(--s)", color: "#2F2A1B", border: "1px solid var(--sand)" }}
                    >
                      {f.year}
                    </div>
                    <div
                      className="mt-1 rounded-xl px-2 py-1 text-sm font-semibold"
                      style={{ background: "var(--s)", color: "#2F2A1B", border: "1px solid var(--sand)" }}
                    >
                      {f.day} {f.month}
                    </div>
                  </div>
                </div>
              </CardHeader>
              <CardContent className="space-y-3">
                <p className="text-sm text-slate-700">{ev.resume}</p>
                <div className="flex items-center gap-2">
                  <a href={ev.lien} className="inline-flex">
                    <Button className="gap-1 btn-primary">
                      <ExternalLink className="h-4 w-4" /> En savoir +
                    </Button>
                  </a>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>
    </section>
  );
}


// ————————————————————————————————————————————
// Revues semestrielles (syntheses originales)
// ————————————————————————————————————————————

function RevuesSemestrielles() {
  const revues = [
    {
      id: "s1-2025",
      titre: "Revue S1 2025",
      periode: "jan - juin 2025",
      resume:
        "Panorama des grandes decisions budgetaires et mesures de marche. Focus sur reformes engrais, tarifs variables cereales et pilotes PSE.",
      points: [
        "Bascules vers ciblage social des subventions intrants",
        "Tarifs variables blé et stock public: stabilisation prix",
        "Pilotes PSE: premiers retours sur MRV et adoption",
      ],
      kpis: [
        { label: "Actus analysees", value: 68 },
        { label: "Pays couverts", value: 22 },
        { label: "Docs primaires", value: 41 },
      ],
      lien: "#",
      pdf: "#",
    },
    {
      id: "s2-2024",
      titre: "Revue S2 2024",
      periode: "juil - dec 2024",
      resume:
        "Tendances de fond: reallocation vers services collectifs et transparence SPM. Etudes d'impact et lecons budgetaires.",
      points: [
        "Hausse budgets services collectifs (conseil, R&D, irrigation)",
        "Debats OMC sur subventions distorsives",
        "Evidences cout-efficacite: achats publics vs transferts cibles",
      ],
      kpis: [
        { label: "Actus analysees", value: 54 },
        { label: "Pays couverts", value: 18 },
        { label: "Docs primaires", value: 29 },
      ],
      lien: "#",
      pdf: "#",
    },
  ];

  return (
    <section className="mt-10" aria-labelledby="revues-title">
      {/* Titre mis en avant */}
      <div className="rounded-2xl bg-gradient-to-r from-[color:var(--p-15)] to-[color:var(--s-20)] p-4 ring-1 ring-[color:var(--sand)]/50 mb-6">
        <div className="flex items-start justify-between gap-4">
          <div>
            <h2 id="revues-title" className="text-2xl md:text-3xl font-extrabold tracking-tight leading-tight">
              <span className="bg-gradient-to-r from-[color:var(--p)] to-[color:var(--s)] bg-clip-text text-transparent">
                Revues semestrielles
              </span>
            </h2>
            <p className="mt-1 text-sm md:text-base text-slate-600">
              Synthèses originales des tendances et décisions marquantes.
            </p>
          </div>
          {/* pas d’actions */}
        </div>
        {/* Barre d’accent */}
        <div className="mt-3 h-1.5 w-28 rounded-full bg-gradient-to-r from-[color:var(--p)] to-[color:var(--s)]" />
      </div>

      {/* Cartes des revues (inchangé) */}
      <div className="grid gap-4 md:grid-cols-2">
        {revues.map((r) => (
          <Card key={r.id} className="rounded-2xl border border-slate-200 shadow-sm">
            <CardHeader className="pb-2">
              <CardTitle className="text-base text-[color:var(--p)]">
                {r.titre} <span className="text-slate-400">• {r.periode}</span>
              </CardTitle>
              <CardDescription>{r.resume}</CardDescription>
            </CardHeader>
            <CardContent>
              <ul className="ml-5 list-disc text-sm text-slate-700 space-y-1">
                {r.points.map((p) => (
                  <li key={p}>{p}</li>
                ))}
              </ul>
              <div className="mt-3 flex flex-wrap gap-2 text-xs">
                {r.kpis.map((k) => (
                  <span
                    key={k.label}
                    className="rounded-full border border-slate-200 bg-slate-50 px-2 py-1"
                  >
                    {k.label}: <span className="font-semibold text-slate-800">{k.value}</span>
                  </span>
                ))}
              </div>
              <div className="mt-4 flex items-center gap-2">
                <a href={r.lien}>
                  <Button className="gap-1 btn-primary">Lire la synthese</Button>
                </a>
                <a href={r.pdf}>
                  <Button className="gap-1 btn-secondary">Telecharger le PDF</Button>
                </a>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </section>
  );
}


// ————————————————————————————————————————————
// Barre de filtres
// ————————————————————————————————————————————

function Filtres({
  query,
  setQuery,
  actifs,
  toggle,
  period,
  setPeriod,
  compact,
  setCompact,
}: {
  query: string;
  setQuery: (s: string) => void;
  actifs: (TypeSoutien | Objectif)[];
  toggle: (t: TypeSoutien | Objectif) => void;
  period: string;
  setPeriod: (p: string) => void;
  compact: boolean;
  setCompact: (b: boolean) => void;
}) {
  const TYPES: { code: TypeSoutien; label: string }[] = [
    { code: "TBP", label: TAG_LABELS.TBP },
    { code: "TBC", label: TAG_LABELS.TBC },
    { code: "SERV", label: TAG_LABELS.SERV },
    { code: "MSM", label: TAG_LABELS.MSM },
  ];
  const OBJECTIFS: { key: Objectif; label: string }[] = [
    { key: "EQUIT", label: TAG_LABELS.EQUIT },
    { key: "PROD", label: TAG_LABELS.PROD },
    { key: "SECAL", label: TAG_LABELS.SECAL },
    { key: "ENV", label: TAG_LABELS.ENV },
  ];

  const typeSelectionne: TypeSoutien | "Tous" =
    (actifs.includes("TBP") && !actifs.includes("TBC") && !actifs.includes("SERV") && !actifs.includes("MSM")) ? "TBP" :
    (!actifs.includes("TBP") && actifs.includes("TBC") && !actifs.includes("SERV") && !actifs.includes("MSM")) ? "TBC" :
    (!actifs.includes("TBP") && !actifs.includes("TBC") && actifs.includes("SERV") && !actifs.includes("MSM")) ? "SERV" :
    (!actifs.includes("TBP") && !actifs.includes("TBC") && !actifs.includes("SERV") && actifs.includes("MSM")) ? "MSM" :
    "Tous";

  function setType(val: TypeSoutien | "Tous") {
    // Retirer tous les types d'abord
    ["TBP","TBC","SERV","MSM"].forEach((t) => {
      if (actifs.includes(t as any)) toggle(t as any);
    });
    if (val !== "Tous") {
      if (!actifs.includes(val)) toggle(val);
    }
  }

  function toggleObjectif(obj: Objectif) {
    toggle(obj);
  }

  return (
    <div className="sticky top-0 z-30 border-b border-slate-200 bg-white/90 backdrop-blur">
      <div className="mx-auto max-w-7xl px-4 py-3">
        <Card className="rounded-2xl border border-slate-200 shadow-sm">
          <CardContent className="pt-4">
            <div className="grid gap-4 md:grid-cols-3">
              {/* Recherche */}
              <div>
                <label className="mb-2 block text-sm font-semibold text-[color:var(--p)]">Recherche</label>
                <div className="relative">
                  <Search className="pointer-events-none absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-slate-400" />
                  <Input
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    placeholder="Instrument, pays, source…"
                    className="w-full pl-9"
                  />
                </div>
                <p className="mt-1 text-[11px] text-slate-500">Exemples : "engrais", "tarifs", "OMC", "Maroc"…</p>
              </div>

              {/* Type de soutien (segmented) */}
              <div>
                <label className="mb-2 block text-sm font-semibold text-[color:var(--p)]">Type de soutien</label>
                <div className="flex flex-wrap gap-2">
                  <Chip active={typeSelectionne === "Tous"} onClick={() => setType("Tous")}>Tous</Chip>
                  {TYPES.map((t) => (
                    <Chip key={t.code} active={typeSelectionne === t.code} onClick={() => setType(t.code)}>
                      {t.label}
                    </Chip>
                  ))}
                </div>
              </div>

              {/* Objectifs (pills) */}
              <div>
                <label className="mb-2 block text-sm font-semibold text-[color:var(--p)]">Objectifs</label>
                <div className="flex flex-wrap gap-2">
                  {OBJECTIFS.map((o) => {
                    const checked = actifs.includes(o.key);
                    return (
                      <button
                        key={o.key}
                        onClick={() => toggleObjectif(o.key)}
                        className={`pill ${checked ? '-obj chip-active' : '-obj'}`}
                      >
                        {TAG_ICON[o.key]} {o.label}
                      </button>
                    );
                  })}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}

// ————————————————————————————————————————————
// Bloc d’intégration Scoop.it (maquette)
// ————————————————————————————————————————————

function ScoopitEmbed() {
  const [modeFictif, setModeFictif] = useState(true);
  const posts = [
    { id: "p1", img: "https://images.unsplash.com/photo-1464226184884-fa280b87c399?q=80&w=1200&auto=format&fit=crop", titre: "Budget engrais 2026 : ciblage par registre social", source: "Gouv.", pays: "Maroc", date: "02 oct. 2025", tags: ["TBP", "PROD"] },
    { id: "p2", img: "https://images.unsplash.com/photo-1500595046743-cd271d694d30?q=80&w=1200&auto=format&fit=crop", titre: "Tarif variable sur le blé pour stabiliser les prix", source: "JO", pays: "Égypte", date: "28 sept. 2025", tags: ["MSM", "SECAL"] },
    { id: "p3", img: "https://images.unsplash.com/photo-1515706886582-54c73c5eaf41?q=80&w=1200&auto=format&fit=crop", titre: "Pilote PSE : pollinisateurs & qualité de l’eau", source: "Agence nat.", pays: "Sénégal", date: "05 oct. 2025", tags: ["SERV", "ENV", "PROD"] },
    { id: "p4", img: "https://images.unsplash.com/photo-1492496913980-501348b61469?q=80&w=1200&auto=format&fit=crop", titre: "Crédits bonifiés pour irrigation efficiente", source: "Banque agri.", pays: "Kenya", date: "30 sept. 2025", tags: ["TBP", "PROD"] },
    { id: "p5", img: "https://upload.wikimedia.org/wikipedia/commons/d/d4/Maize_bags.jpg", titre: "Achats publics : stock stratégique maïs", source: "Agence stock", pays: "Ghana", date: "01 oct. 2025", tags: ["TBP", "SECAL"] },
    { id: "p6", img: "https://images.unsplash.com/photo-1519681393784-d120267933ba?q=80&w=1200&auto=format&fit=crop", titre: "OMC : discipline des subventions distorsives", source: "OMC", pays: "Monde", date: "06 oct. 2025", tags: ["MSM", "SECAL"] },
  ];

  return (
    <section aria-label="Flux de veille" aria-labelledby="scoopit-title" className="mt-10">
      {/* Titre mis en avant (même style que ta section Flux local) */}
      <div className="rounded-2xl bg-gradient-to-r from-[color:var(--p-15)] to-[color:var(--s-20)] p-4 ring-1 ring-[color:var(--sand)]/50 mb-6">
        <div className="flex items-start justify-between gap-4">
          <div>
            <h2 id="scoopit-title" className="text-2xl md:text-3xl font-extrabold tracking-tight leading-tight">
              <span className="bg-gradient-to-r from-[color:var(--p)] to-[color:var(--s)] bg-clip-text text-transparent">
                Flux d’actualité (Scoop.it)
              </span>
            </h2>
            <p className="mt-1 text-sm md:text-base text-slate-600">
              Sélection automatiquement agrégée depuis notre curation.
            </p>
          </div>
          {/* (pas d’actions) */}
        </div>
        {/* Barre d’accent */}
        <div className="mt-3 h-1.5 w-28 rounded-full bg-gradient-to-r from-[color:var(--p)] to-[color:var(--s)]" />
      </div>

      {modeFictif ? (
        <div>
          <div className="grid gap-4 md:grid-cols-3">
            {posts.map((p) => (
              <Card key={p.id} className="overflow-hidden rounded-2xl border border-slate-200 shadow-sm">
                <div className="h-36 w-full overflow-hidden">
                  <img
                    src={p.img}
                    alt="visuel"
                    className="h-full w-full object-cover"
                    onError={(e) => {
                      const svg = `<svg xmlns='http://www.w3.org/2000/svg' width='1200' height='400'>
  <defs><linearGradient id='g' x1='0' y1='0' x2='1' y2='1'><stop offset='0%' stop-color='#e2e8f0'/><stop offset='100%' stop-color='#f8fafc'/></linearGradient></defs>
  <rect width='100%' height='100%' fill='url(#g)'/>
  <text x='50%' y='50%' dominant-baseline='middle' text-anchor='middle' fill='#334155' font-family='Arial, Helvetica, sans-serif' font-size='28'>Image non disponible</text>
</svg>`;
                      (e.currentTarget as HTMLImageElement).src = `data:image/svg+xml;utf8,${encodeURIComponent(svg)}`;
                      (e.currentTarget as HTMLImageElement).onerror = null;
                    }}
                  />
                </div>
                <CardHeader className="pb-2">
                  <CardTitle className="text-base leading-snug line-clamp-2">{p.titre}</CardTitle>
                  <CardDescription className="flex items-center gap-2 text-xs">
                    <span className="rounded bg-slate-100 px-1.5 py-0.5">{p.source}</span>
                    <span>•</span>
                    <span>{p.pays}</span>
                    <span>•</span>
                    <span>{p.date}</span>
                  </CardDescription>
                </CardHeader>
                <CardContent className="pb-4">
                  <div className="flex flex-wrap gap-1.5">
                    {p.tags.map((t) =>
                      (TAG_LABELS as any)[t] ? (
                        <TagPill key={t as string} tag={t as any} />
                      ) : (
                        <Badge key={t as string} variant="secondary">
                          {t as string}
                        </Badge>
                      )
                    )}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
          <div className="mt-4 flex items-center justify-center gap-2">
            <Button className="rounded-full btn-secondary">Précédent</Button>
            <div className="text-xs text-[color:var(--p)]">Page 1 / 5</div>
            <Button className="rounded-full btn-primary">Suivant</Button>
          </div>
        </div>
      ) : (
        <div className="aspect-[16/9] w-full overflow-hidden rounded-2xl border border-slate-200">
          <iframe
            src={"EMBED_URL_SCOOPIT"}
            title="Scoop.it — Veille FARM"
            className="h-full w-full"
            loading="lazy"
          />
        </div>
      )}
    </section>
  );
}

// ————————————————————————————————————————————
// Page principale
// ————————————————————————————————————————————

export default function MaquetteVeilleFARM() {
  const [query, setQuery] = useState("");
  const [period, setPeriod] = useState("30");
  const [compact, setCompact] = useState(false);
  const [actifs, setActifs] = useState<(TypeSoutien | Objectif)[]>(["TBP", "MSM", "ENV"]);

  const toggle = (t: TypeSoutien | Objectif) =>
    setActifs((prev) => (prev.includes(t) ? prev.filter((x) => x !== t) : [...prev, t]));

  const filtresActifs = useMemo(() => new Set(actifs), [actifs]);
  const data = useMemo(() => {
    return MOCK.filter((c) => {
      const hasTag = [c.type, ...c.objectifs].some((t) => filtresActifs.has(t));
      const matchQuery = query
        ? [c.titre, c.pays, c.region, c.instrument, c.source].some((s) =>
            s.toLowerCase().includes(query.toLowerCase())
          )
        : true;
      return hasTag && matchQuery;
    });
  }, [actifs, query]);

  return (
    <div className="min-h-screen bg-white">
      {/* Palette FARM inspirée de la page Observatoire */}
      <style>{`:root{--p:#0B5340;        /* forêt */
  --s:#F2D788;        /* miel */
  --bg:#F6F5EF;
  --ink:#0F172A;
  --muted:#6B7280;
  --sand:#E3C46C;
  --p-15:rgba(11,83,64,.15);
  --p-08:rgba(11,83,64,.08);
  --s-20:rgba(242,215,136,.35);
  --card:#FFFFFF;}
/* Buttons */
.btn-primary{background:var(--p);color:#fff;border:1px solid var(--p);border-radius:0.75rem;padding:.5rem .75rem}
.btn-primary:hover{filter:brightness(.95)}
.btn-secondary{background:var(--s);color:#2F2A1B;border:1px solid var(--sand);border-radius:0.75rem;padding:.5rem .75rem}
.btn-secondary:hover{filter:brightness(.97)}
/* Links */
.link-primary{color:var(--p)}
.link-primary:hover{text-decoration:underline}
/* Chips */
.chip{border:1px solid #CBD5E1;background:#fff;color:#334155;border-radius:9999px}
.chip-active{background:var(--p);border-color:var(--p);color:#fff;border-radius:9999px}
/* Pills */
.pill{display:inline-flex;align-items:center;gap:.25rem;border-radius:9999px;padding:.25rem .5rem;font-size:.75rem;border:1px solid transparent}
.pill.-tbp{background:var(--p);color:#fff;border-color:var(--p)}
.pill.-tbc{background:#5B4B00;color:#fff;border-color:#5B4B00}
.pill.-serv{background:var(--p);color:#fff;border-color:var(--p)}
.pill.-msm{background:var(--s);color:#2F2A1B;border:1px solid var(--sand)}
.pill.-obj{background:var(--p-08);color:var(--p)}
/* Cards */
.card-accent{box-shadow:inset 0 3px 0 0 var(--p-08)}
/* Headings */
.h-underline{position:relative}
.h-underline:after{content:"";position:absolute;left:0;bottom:-6px;height:3px;width:64px;background:linear-gradient(90deg,var(--p) 0%, var(--s) 100%);border-radius:2px}`}</style>
      {/* En-tête */}
      <header className="relative border-b border-slate-200" style={{ background: "var(--card)" }}>
        {/* Fond décoratif discret */}
        <div
        className="pointer-events-none absolute inset-0 opacity-70"
        style={{
          background: `
            radial-gradient(1000px 360px at 100% -10%, var(--p-15), transparent),
            radial-gradient(800px 300px at 92% 0%, transparent, transparent)
          `
        }}
      />

  <div className="relative mx-auto max-w-7xl px-4 py-8">
    {/* Titre + actions */}
    <div className="flex flex-col items-start justify-between gap-4 sm:flex-row sm:items-center">
      <div>
        <h1 className="text-3xl md:text-4xl font-extrabold tracking-tight leading-tight">
          <span className="bg-[linear-gradient(90deg,_var(--p)_0%,_var(--p)_100%,_var(--s)_100%)] bg-clip-text text-transparent">
            Veille — Soutiens publics à l’agriculture
          </span>
        </h1>

        <p className="mt-2 max-w-3xl text-sm md:text-base text-slate-600">
          L’essentiel de l’actualité des soutiens publics à l’agriculture, organisé par type de soutien et objectifs, avec
          un décryptage rapide de ce qui évolue, qui est concerné et quels impacts.
        </p>
        {/* Barre d’accent */}
        <div className="mt-4 h-1.5 w-32 rounded-full bg-[linear-gradient(90deg,_var(--p)_0%,_var(--p)_100%,_var(--s)_100%)]" />

      </div>

      <div className="flex w-full flex-col gap-2 sm:w-auto sm:flex-row">
        <Button className="btn-secondary w-full sm:w-auto">Proposer une source</Button>
        <Button className="btn-primary w-full sm:w-auto">S’abonner à la newsletter</Button>
      </div>
    </div>
  </div>
</header>



      {/* Filtres */}
      <Filtres
        query={query}
        setQuery={setQuery}
        actifs={actifs}
        toggle={toggle}
        period={period}
        setPeriod={setPeriod}
        compact={compact}
        setCompact={setCompact}
      />

      {/* Corps */}
      <main className="mx-auto max-w-7xl px-4 py-8">
        {/* Flux local (maquette de cartes) */}
        <section className="mt-0" aria-labelledby="flux-local-title">
          {/* Bandeau d’en-tête mis en avant */}
          <div className="rounded-2xl bg-gradient-to-r from-[color:var(--p-15)] to-[color:var(--s-20)] p-4 ring-1 ring-[color:var(--sand)]/50 mb-6">
            <div className="flex items-start justify-between gap-4">
              <div>
                <h2 id="flux-local-title" className="text-2xl md:text-3xl font-extrabold tracking-tight leading-tight">
                  <span className="bg-[linear-gradient(90deg,_var(--p)_0%,_var(--p)_50%,_var(--s)_100%)] bg-clip-text text-transparent">
                    Le décryptage FARM
                  </span>
                </h2>
                <p className="mt-1 text-sm md:text-base text-slate-600">
                  Notre selection d'article.
                </p>
              </div>
              {/* (actions supprimées) */}
            </div>
            {/* Barre d’accent */}
            <div className="mt-4 h-1.5 w-32 rounded-full bg-[linear-gradient(90deg,_var(--p)_0%,_var(--p)_50%,_var(--s)_100%)]" />

          </div>

          {/* Grille des cartes */}
          <div className={`grid gap-4 ${compact ? "md:grid-cols-3" : "md:grid-cols-2"}`}>
            {data.map((c) => (
              <Carte key={c.id} c={c} />
            ))}
          </div>

          {data.length === 0 && (
            <div className="mt-6 rounded-2xl border border-slate-200 bg-white p-6 text-sm text-slate-600">
              Aucune carte ne correspond à vos filtres. Essayez d’élargir la recherche.
            </div>
          )}
        </section>


         {/* Embed Scoop.it */}
        <ScoopitEmbed />

        {/* Événements & grands rendez-vous */}
        <EvenementsRendezVous />

        {/* Revues semestrielles */}
        <RevuesSemestrielles />
      </main>

      {/* Pied de page */}
      <footer className="mt-10 border-t border-slate-200" style={{background:"var(--card)"}}>
        <div className="mx-auto max-w-7xl px-4 py-6 text-xs text-slate-500">
          © {new Date().getFullYear()} Fondation FARM — Observatoire mondial des soutiens publics à l’agriculture • Glossaire • Mentions légales
        </div>
      </footer>
    </div>
  );
}
